/* global wp, sfsb_params, accounting */
jQuery( function( $ ) {

	/**
	 * Legacy AJAX Support
	 * Force a fragment refresh when the cart is updated via classic AJAX.
	 */
	$( document.body ).on( 'updated_wc_div updated_checkout', function() {
		$( document.body ).trigger( 'wc_fragment_refresh' );
	} );

    /**
     * Store API / Blocks Support
     * Listens to changes in wc/store/cart data store.
     */
    const initStoreApiListener = () => {
        if ( typeof wp === 'undefined' || ! wp.data || ! wp.data.subscribe || ! wp.data.select ) {
            return;
        }

        const namespace = 'wc/store/cart';
        
        // Helper function to safely get store
        const getStore = () => {
            try {
                return wp.data.select( namespace );
            } catch ( err ) {
                return null;
            }
        };

        // Don't proceed if store doesn't exist (yet)
        if ( ! getStore() ) {
            return;
        }

        // Subscribe to store changes
        // We use a flag to prevent multiple subscriptions if this runs multiple times
        if ( window.sfsb_subscribed ) return;
        
        wp.data.subscribe( () => {
            const store = getStore();
            if ( ! store ) return;

            const cartData = store.getCartData();
            updateBarBlock( cartData );
        } );
        
        window.sfsb_subscribed = true;
    };

    /**
     * Update the bar based on Store API cart data
     */
    function updateBarBlock( cartData ) {
        if ( typeof sfsb_params === 'undefined' || ! cartData || ! cartData.totals ) {
            return;
        }

        const params = sfsb_params;
        const decimals = parseInt( params.decimals, 10 );
        const divisor = Math.pow( 10, decimals );
        const totals = cartData.totals;

        // Calculate Cart Content Total
        // Store API returns string integers (minor units)
        let totalCents = parseInt( totals.total_items, 10 );

        // Add tax if configured to display inclusive
        if ( params.incl_tax == '1' || params.incl_tax === true ) {
            totalCents += parseInt( totals.total_items_tax, 10 );
        }

        const total = totalCents / divisor;
        const threshold = parseFloat( params.threshold );
        const missing = threshold - total;

        let message = '';
        
        // Helper to format currency
        const format = ( val ) => {
             if ( typeof accounting !== 'undefined' && typeof accounting.formatMoney === 'function' ) {
                 return accounting.formatMoney( val, {
                     symbol: params.currency_symbol,
                     precision: decimals
                 } );
             }
             return params.currency_symbol + val.toFixed( decimals );
        };

        if ( total <= 0 ) {
            message = params.msg_initial.replace( '{amount}', format( threshold ) );
        } else if ( missing > 0 ) {
            message = params.msg_progress.replace( '{amount}', format( missing ) );
        } else {
            message = params.msg_completed;
        }

        // Update DOM
        const $bar = $( '#sfsb-bar' );
        if ( $bar.length ) {
            $bar.find( '.sfsb-message' ).html( message );
        }
    }

    // Attempt to init immediately, then again on load to ensure store is registered
    initStoreApiListener();
    $( window ).on( 'load', initStoreApiListener );

    /**
     * Intercept fetch requests to detect Store API add-to-cart (when not handled by Blocks)
     */
    const originalFetch = window.fetch;
    window.fetch = function( ...args ) {
        const [ resource ] = args;
        return originalFetch.apply( this, args ).then( response => {
            if ( typeof resource === 'string' && (
                resource.includes( 'wc/store/v1/cart/add-item' ) || 
                resource.includes( 'wc/store/v1/cart/update-item' ) || 
                resource.includes( 'wc/store/v1/cart/remove-item' )
            ) ) {
                // If Item added via Store API, force refresh of the Block Data Store
                if ( response.ok && typeof wp !== 'undefined' && wp.data && wp.data.dispatch ) {
                    try {
                        wp.data.dispatch( 'wc/store/cart' ).invalidateResolution( 'getCartData' );
                    } catch( err ) {
                        // Fallback: Trigger legacy refresh just in case
                        $( document.body ).trigger( 'wc_fragment_refresh' );
                    }
                }
            }
            return response;
        } );
    };

} );
