<?php
/**
 * Plugin Name: Simple Free Shipping Bar for WooCommerce
 * Description: A sticky notification bar to encourage users to add more products to cart for free shipping.
 * Version: 1.0.2
 * Author:      Ocma Labs
 * Author URI:  https://ocmalabs.com
 * Text Domain: simple-free-shipping-bar-ocma
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

// Define Pro URL
if ( ! defined( 'SFSB_PRO_URL' ) ) {
    define( 'SFSB_PRO_URL', 'https://ocmalabs.com/simple-free-shipping-bar' ); 
}

class Simple_Free_Shipping_Bar {

    /**
     * Instance of this class.
     *
     * @var Simple_Free_Shipping_Bar
     */
    private static $instance = null;

    /**
     * Get the singleton instance.
     *
     * @return Simple_Free_Shipping_Bar
     */
    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor.
     */
    private function __construct() {
        // Initialize Admin Settings
        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );

        // Footer Text
        add_filter( 'admin_footer_text', array( $this, 'update_admin_footer' ) );

        // Frontend Hook
        add_action( 'wp_footer', array( $this, 'render_bar' ) );
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
        // Ajax Fragments Hook
        add_filter( 'woocommerce_add_to_cart_fragments', array( $this, 'update_fragments' ) );
        // Ensure fragments cache is invalidated when settings change
        add_filter( 'woocommerce_add_to_cart_hash', array( $this, 'add_settings_to_cart_hash' ) );
        // Shortcode
        add_shortcode( 'simple_free_shipping_bar', array( $this, 'render_shortcode' ) );
    }

    /**
     * Add settings version to cart hash to invalidate fragments cache on settings change.
     *
     * @param string $hash
     * @return string
     */
    public function add_settings_to_cart_hash( $hash ) {
        // Append relevant settings to the hash
        $hash .= get_option( 'sfsb_position', 'top' );
        $hash .= get_option( 'sfsb_display_method', 'automatic' );
        $hash .= get_option( 'sfsb_threshold', 50 );
        $hash .= get_option( 'sfsb_bg_color' );
        $hash .= get_option( 'sfsb_bg_color_progress' );  // New
        $hash .= get_option( 'sfsb_bg_color_completed' ); // New
        return $hash;
    }

    /**
     * Enqueue frontend scripts.
     */
    public function enqueue_scripts() {
        wp_enqueue_script(
            'sfsb-frontend',
            plugin_dir_url(__FILE__) . 'assets/js/sfsb-frontend.js',
            array('jquery', 'wc-cart-fragments', 'wp-data', 'accounting'),
            filemtime(plugin_dir_path(__FILE__) . 'assets/js/sfsb-frontend.js'),
            true
        );

        // Localize parameters for Store API / Blocks support
        $threshold = (float) get_option( 'sfsb_threshold', 50 );
        $params = array(
            'threshold'       => $threshold,
            'msg_initial'     => get_option( 'sfsb_msg_initial', __( 'Free shipping for orders over {amount}', 'simple-free-shipping-bar-ocma' ) ),
            'msg_progress'    => get_option( 'sfsb_msg_progress', __( 'Buy {amount} more to get free shipping!', 'simple-free-shipping-bar-ocma' ) ),
            'msg_completed'   => get_option( 'sfsb_msg_completed', __( 'You\'ve got free shipping!', 'simple-free-shipping-bar-ocma' ) ),
            'currency_symbol' => function_exists( 'get_woocommerce_currency_symbol' ) ? get_woocommerce_currency_symbol() : '$',
            'decimals'        => function_exists( 'wc_get_price_decimals' ) ? wc_get_price_decimals() : 2,
            'incl_tax'        => function_exists( 'WC' ) && WC()->cart ? WC()->cart->display_prices_including_tax() : false,
        );
        wp_localize_script( 'sfsb-frontend', 'sfsb_params', $params );
    }

    /**
     * Enqueue Admin scripts.
     */
    public function enqueue_admin_scripts( $hook ) {
        // Only load on our settings page
        if ( 'settings_page_simple-free-shipping-bar-ocma' !== $hook ) {
            return;
        }
        
        wp_enqueue_style(
            'sfsb-admin-css',
            plugin_dir_url( __FILE__ ) . 'assets/css/sfsb-admin.css',
            array(),
            filemtime( plugin_dir_path( __FILE__ ) . 'assets/css/sfsb-admin.css' )
        );
    }

    /**
     * Custom Footer Text.
     */
    public function update_admin_footer( $text ) {
        $screen = get_current_screen();
        if ( $screen && 'settings_page_simple-free-shipping-bar-ocma' === $screen->id ) {
            /* translators: %s: Author link */
            $text = sprintf( __( 'Thank you for choosing Simple Free Shipping Bar. Built with ❤️ by %s.', 'simple-free-shipping-bar-ocma' ), '<a href="https://ocmalabs.com" target="_blank">Ocma Labs</a>' );
        }
        return $text;
    }

    /**
     * Add options page to the Settings menu.
     */
    public function add_admin_menu() {
        add_options_page(
            __( 'Simple Free Shipping Bar', 'simple-free-shipping-bar-ocma' ),
            __( 'Free Shipping Bar', 'simple-free-shipping-bar-ocma' ),
            'manage_options',
            'simple-free-shipping-bar-ocma',
            array( $this, 'options_page_html' )
        );
    }

    /**
     * Register settings and fields.
     */
    public function register_settings() {
        $fields = array(
            // Enabled
            'sfsb_enabled' => array(
                'label'     => __( 'Enable Bar', 'simple-free-shipping-bar-ocma' ),
                'callback'  => 'render_checkbox_field',
                'section'   => 'sfsb_section_config',
                'page'      => 'simple-free-shipping-bar-ocma',
                'args'      => array( 'label_for' => 'sfsb_enabled' ),
                'register'  => array() // No sanitize callback needed for boolean
            ),
            // Threshold
            'sfsb_threshold' => array(
                'label'     => __( 'Free Shipping Amount', 'simple-free-shipping-bar-ocma' ),
                'callback'  => 'render_text_field',
                'section'   => 'sfsb_section_config',
                'page'      => 'simple-free-shipping-bar-ocma',
                'args'      => array( 'label_for' => 'sfsb_threshold', 'type' => 'number', 'desc' => 'Enter the amount required for free shipping.' ),
                'register'  => array( 'type' => 'number', 'sanitize_callback' => 'absint', 'default' => 50 )
            ),
            // Messages
            'sfsb_msg_initial' => array(
                'label'     => __( 'Initial Message', 'simple-free-shipping-bar-ocma' ),
                'callback'  => 'render_text_field',
                'section'   => 'sfsb_section_messages',
                'page'      => 'simple-free-shipping-bar-ocma',
                'args'      => array( 'label_for' => 'sfsb_msg_initial', 'desc' => 'Shown when cart is empty.' ),
                'register'  => array( 'default' => __( 'Free shipping for orders over {amount}', 'simple-free-shipping-bar-ocma' ), 'sanitize_callback' => 'sanitize_text_field' )
            ),
            'sfsb_msg_progress' => array(
                'label'     => __( 'Progress Message', 'simple-free-shipping-bar-ocma' ),
                'callback'  => 'render_text_field',
                'section'   => 'sfsb_section_messages',
                'page'      => 'simple-free-shipping-bar-ocma',
                'args'      => array( 'label_for' => 'sfsb_msg_progress', 'desc' => 'Use {amount} to show remaining amount.' ),
                'register'  => array( 'default' => __( 'Buy {amount} more to get free shipping!', 'simple-free-shipping-bar-ocma' ), 'sanitize_callback' => 'sanitize_text_field' )
            ),
            'sfsb_msg_completed' => array(
                'label'     => __( 'Completed Message', 'simple-free-shipping-bar-ocma' ),
                'callback'  => 'render_text_field',
                'section'   => 'sfsb_section_messages',
                'page'      => 'simple-free-shipping-bar-ocma',
                'args'      => array( 'label_for' => 'sfsb_msg_completed', 'desc' => 'Shown when free shipping is reached.' ),
                'register'  => array( 'default' => __( 'You\'ve got free shipping!', 'simple-free-shipping-bar-ocma' ), 'sanitize_callback' => 'sanitize_text_field' )
            ),
            // Colors - Initial
            'sfsb_bg_color' => array(
                'label'     => __( 'Initial Background Color', 'simple-free-shipping-bar-ocma' ),
                'callback'  => 'render_color_field',
                'section'   => 'sfsb_section_appearance',
                'page'      => 'simple-free-shipping-bar-ocma',
                'args'      => array( 'label_for' => 'sfsb_bg_color', 'desc' => 'Color when cart is empty.' ),
                'register'  => array( 'default' => '#64748b', 'sanitize_callback' => 'sanitize_hex_color' )
            ),
            // Colors - Progress
            'sfsb_bg_color_progress' => array(
                'label'     => __( 'Progress Background Color', 'simple-free-shipping-bar-ocma' ),
                'callback'  => 'render_color_field',
                'section'   => 'sfsb_section_appearance',
                'page'      => 'simple-free-shipping-bar-ocma',
                'args'      => array( 'label_for' => 'sfsb_bg_color_progress', 'desc' => 'Color when user is adding products.' ),
                'register'  => array( 'default' => '#4f46e5', 'sanitize_callback' => 'sanitize_hex_color' )
            ),
            // Colors - Completed
            'sfsb_bg_color_completed' => array(
                'label'     => __( 'Completed Background Color', 'simple-free-shipping-bar-ocma' ),
                'callback'  => 'render_color_field',
                'section'   => 'sfsb_section_appearance',
                'page'      => 'simple-free-shipping-bar-ocma',
                'args'      => array( 'label_for' => 'sfsb_bg_color_completed', 'desc' => 'Color when free shipping is reached.' ),
                'register'  => array( 'default' => '#22c55e', 'sanitize_callback' => 'sanitize_hex_color' )
            ),
            // Text Color
            'sfsb_text_color' => array(
                'label'     => __( 'Text Color', 'simple-free-shipping-bar-ocma' ),
                'callback'  => 'render_color_field',
                'section'   => 'sfsb_section_appearance',
                'page'      => 'simple-free-shipping-bar-ocma',
                'args'      => array( 'label_for' => 'sfsb_text_color' ),
                'register'  => array( 'default' => '#ffffff', 'sanitize_callback' => 'sanitize_hex_color' )
            ),
            // Display Options
            'sfsb_display_method' => array(
                'label'     => __( 'Display Method', 'simple-free-shipping-bar-ocma' ),
                'callback'  => 'render_select_field',
                'section'   => 'sfsb_section_display',
                'page'      => 'simple-free-shipping-bar-ocma',
                'args'      => array( 
                    'label_for' => 'sfsb_display_method',
                    'options'   => array(
                        'automatic' => __( 'Automatic (Sticky)', 'simple-free-shipping-bar-ocma' ),
                        'shortcode' => __( 'Shortcode Placement', 'simple-free-shipping-bar-ocma' ),
                    ),
                    'default'   => 'automatic',
                    'desc'      => __( 'Use [simple_free_shipping_bar] if Shortcode is selected.', 'simple-free-shipping-bar-ocma' )
                ),
                'register'  => array( 'default' => 'automatic', 'sanitize_callback' => 'sanitize_key' )
            ),
            'sfsb_position' => array(
                'label'     => __( 'Position (Automatic)', 'simple-free-shipping-bar-ocma' ),
                'callback'  => 'render_select_field',
                'section'   => 'sfsb_section_display',
                'page'      => 'simple-free-shipping-bar-ocma',
                'args'      => array( 
                    'label_for' => 'sfsb_position',
                    'options'   => array(
                        'top'    => __( 'Top', 'simple-free-shipping-bar-ocma' ),
                        'bottom' => __( 'Bottom', 'simple-free-shipping-bar-ocma' ),
                    ),
                    'default'   => 'top',
                    'desc'      => __( 'Applies only when Display Method is Automatic.', 'simple-free-shipping-bar-ocma' )
                ),
                'register'  => array( 'default' => 'top', 'sanitize_callback' => 'sanitize_key' )
            ),
            // Premium Placeholders (Removed from Free version)
        );

        // Allow Add-ons to extend/modify fields
        $fields = apply_filters( 'sfsb_settings_fields', $fields );

        // Register Sections
        add_settings_section(
            'sfsb_section_config',
            __( 'Configuration', 'simple-free-shipping-bar-ocma' ),
            null,
            'simple-free-shipping-bar-ocma'
        );
        add_settings_section(
            'sfsb_section_messages',
            __( 'Messages', 'simple-free-shipping-bar-ocma' ),
            null,
            'simple-free-shipping-bar-ocma'
        );
        add_settings_section(
            'sfsb_section_appearance',
            __( 'Appearance', 'simple-free-shipping-bar-ocma' ),
            null,
            'simple-free-shipping-bar-ocma'
        );
        add_settings_section(
            'sfsb_section_display',
            __( 'Display', 'simple-free-shipping-bar-ocma' ),
            null,
            'simple-free-shipping-bar-ocma'
        );

        // Loop through fields to register them
        foreach ( $fields as $id => $field ) {
            register_setting( 'sfsb_settings_group', $id, $field['register'] );
            add_settings_field(
                $id,
                $field['label'],
                array( $this, $field['callback'] ),
                $field['page'],
                $field['section'],
                $field['args']
            );
        }
    }

    /**
     * Render the Settings Page HTML with Tabs.
     */
    public function options_page_html() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $active_tab = isset( $_GET['tab'] ) ? sanitize_key( $_GET['tab'] ) : 'general';

        $tabs = array(
            'general' => __( 'General Settings', 'simple-free-shipping-bar-ocma' ),
            'premium' => __( 'Premium Features', 'simple-free-shipping-bar-ocma' ),
        );
        $tabs = apply_filters( 'sfsb_settings_tabs', $tabs );
        ?>
        <div class="wrap">
            <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
            
            <?php 
            // settings_errors(); // Prevent duplicate notices as WP might display them automatically
            ?>

            <?php $this->render_pro_banner(); ?>

            <h2 class="nav-tab-wrapper">
                <?php foreach ( $tabs as $tab_id => $tab_label ) : ?>
                    <a href="?page=simple-free-shipping-bar-ocma&tab=<?php echo esc_attr( $tab_id ); ?>" class="nav-tab <?php echo $active_tab == $tab_id ? 'nav-tab-active' : ''; ?>"><?php echo esc_html( $tab_label ); ?></a>
                <?php endforeach; ?>
            </h2>
            
            <?php
            if ( 'general' === $active_tab ) {
                echo '<form action="options.php" method="post" class="sfsb-wrap">';
                settings_fields( 'sfsb_settings_group' );

                // Card Layout
                $sections = array(
                    'sfsb_section_config'     => __( 'Configuration', 'simple-free-shipping-bar-ocma' ),
                    'sfsb_section_messages'   => __( 'Messages', 'simple-free-shipping-bar-ocma' ),
                    'sfsb_section_appearance' => __( 'Appearance', 'simple-free-shipping-bar-ocma' ),
                    'sfsb_section_display'    => __( 'Display', 'simple-free-shipping-bar-ocma' ),
                );

                foreach ( $sections as $section_id => $section_title ) {
                    echo '<div class="sfsb-card">';
                    echo '<div class="sfsb-card-header"><h2>' . esc_html( $section_title ) . '</h2></div>';
                    echo '<div class="sfsb-card-body">';
                    echo '<table class="form-table" role="presentation">';
                    do_settings_fields( 'simple-free-shipping-bar-ocma', $section_id );
                    echo '</table>';
                    echo '</div>'; // card-body
                    echo '</div>'; // card
                }

                echo '<div class="sfsb-save-bar">';
                submit_button( __( 'Save Settings', 'simple-free-shipping-bar-ocma' ), 'primary', 'submit', false );
                echo '</div>';
                
                echo '</form>';
            } elseif ( 'premium' === $active_tab ) {
                $is_premium_active = apply_filters( 'sfsb_is_premium_active', false );
                if ( $is_premium_active ) {
                    // Hook for Pro plugin to render its content
                    do_action( 'sfsb_render_premium_tab_content' );
                } else {
                    $this->render_premium_upsell_page();
                }
            } else {
                do_action( 'sfsb_settings_tab_content_' . $active_tab );
            }            
            
            ?>
        </div>
        <?php
    }

    /**
     * Render Premium Section Description. (Removed)
     */
    public function render_premium_section_desc() {}

    /**
     * Render Premium Field (Removed)
     * @param array $args Field arguments.
     */
    public function render_premium_field( $args ) {}

    /**
     * Render Checkbox Field.
     *
     * @param array $args Field arguments.
     */
    public function render_checkbox_field( $args ) {
        $option = get_option( $args['label_for'] );
        ?>
        <input type="checkbox" id="<?php echo esc_attr( $args['label_for'] ); ?>" name="<?php echo esc_attr( $args['label_for'] ); ?>" value="1" <?php checked( 1, $option, true ); ?> />
        <?php
    }

    /**
     * Render Text Field.
     *
     * @param array $args Field arguments.
     */
    public function render_text_field( $args ) {
        $option = get_option( $args['label_for'] );
        $type = isset( $args['type'] ) ? $args['type'] : 'text';
        $desc = isset( $args['desc'] ) ? $args['desc'] : '';
        ?>
        <input type="<?php echo esc_attr( $type ); ?>" id="<?php echo esc_attr( $args['label_for'] ); ?>" name="<?php echo esc_attr( $args['label_for'] ); ?>" value="<?php echo esc_attr( $option ); ?>" class="regular-text" />
        <?php if ( $desc ) : ?>
            <p class="description"><?php echo esc_html( $desc ); ?></p>
        <?php endif;
    }

    /**
     * Render Color Picker Field.
     *
     * @param array $args Field arguments.
     */
    public function render_color_field( $args ) {
        $option = get_option( $args['label_for'] );
        // Fallback defaults if empty
        if ( empty( $option ) ) {
            $default = '#ffffff';
            if ( $args['label_for'] === 'sfsb_bg_color' ) $default = '#64748b';
            if ( $args['label_for'] === 'sfsb_bg_color_progress' ) $default = '#4f46e5';
            if ( $args['label_for'] === 'sfsb_bg_color_completed' ) $default = '#22c55e';
            $option = $default;
        }
        ?>
        <input type="color" id="<?php echo esc_attr( $args['label_for'] ); ?>" name="<?php echo esc_attr( $args['label_for'] ); ?>" value="<?php echo esc_attr( $option ); ?>" />
        <?php
    }

    /**
     * Render Select Field.
     * * @param array $args Field arguments.
     */
    public function render_select_field( $args ) {
        $option = get_option( $args['label_for'] );
        // Use default if not set
        if ( empty( $option ) && isset( $args['default'] ) ) {
            $option = $args['default'];
        }
        
        $options = isset( $args['options'] ) ? $args['options'] : array();
        ?>
        <select id="<?php echo esc_attr( $args['label_for'] ); ?>" name="<?php echo esc_attr( $args['label_for'] ); ?>">
            <?php foreach ( $options as $value => $label ) : ?>
                <option value="<?php echo esc_attr( $value ); ?>" <?php selected( $value, $option ); ?>>
                    <?php echo esc_html( $label ); ?>
                </option>
            <?php endforeach; ?>
        </select>
        <?php if ( ! empty( $args['desc'] ) ) : ?>
            <p class="description"><?php echo esc_html( $args['desc'] ); ?></p>
        <?php endif;
    }

    /**
     * Render Shortcode.
     */
    public function render_shortcode() {
        return $this->get_bar_html( true );
    }

    /**
     * Get the logic and HTML for the bar.
     *
     * @param bool $is_shortcode Whether it's called from a shortcode.
     * @return string HTML content of the bar.
     */
    public function get_bar_html( $is_shortcode = false ) {
        // Check if enabled
        if ( ! get_option( 'sfsb_enabled' ) ) {
            return '';
        }

        // Allow Add-ons to disable logic
        if ( ! apply_filters( 'sfsb_should_show_bar', true ) ) {
            return '';
        }

        // Ensure WooCommerce is active and cart exists
        if ( ! function_exists( 'WC' ) || ! WC()->cart ) {
            return '';
        }

        // Get settings
        $threshold      = (float) get_option( 'sfsb_threshold', 50 );
        $msg_initial    = get_option( 'sfsb_msg_initial', __( 'Free shipping for orders over {amount}', 'simple-free-shipping-bar-ocma' ) );
        $msg_progress   = get_option( 'sfsb_msg_progress', __( 'Buy {amount} more to get free shipping!', 'simple-free-shipping-bar-ocma' ) );
        $msg_completed  = get_option( 'sfsb_msg_completed', __( 'You\'ve got free shipping!', 'simple-free-shipping-bar-ocma' ) );
        
        $bg_initial     = get_option( 'sfsb_bg_color', '#64748b' );
        $bg_progress    = get_option( 'sfsb_bg_color_progress', '#4f46e5' );
        $bg_completed   = get_option( 'sfsb_bg_color_completed', '#22c55e' );
        $text_color     = get_option( 'sfsb_text_color', '#ffffff' );
        
        $display_method = get_option( 'sfsb_display_method', 'automatic' );
        $position       = get_option( 'sfsb_position', 'top' );

        // Calculate total
        if ( WC()->cart->display_prices_including_tax() ) {
            $total = WC()->cart->get_cart_contents_total() + WC()->cart->get_cart_contents_tax();
        } else {
            $total = WC()->cart->get_cart_contents_total();
        }

        $missing = $threshold - $total;

        // --- VISUAL IMPROVEMENTS (Icons & Styles) ---
        
        // 1. Estilo para el precio: Negrita y subrayado sutil
        $style_amount = 'font-weight: 700; border-bottom: 1px solid rgba(255,255,255,0.4); padding-bottom: 0px; margin: 0 5px; display: inline-block;';
        
        // 2. Definimos los Iconos SVG (Ligeros y nítidos)
        // Nota: wp_kses_post elimina SVGs, así que imprimiremos esto sin escapar al final (es seguro porque está hardcodeado aquí)
        
        // Icono Fuego (Aparece mientras compras)
        $icon_fire = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512" style="display:inline-block; width:1em; height:1em; vertical-align:-0.125em; fill:currentColor; margin-left:6px; opacity:0.9;"><path d="M159.3 5.4c7.8-7.3 19.9-7.2 27.7 .1c27.6 25.9 53.5 53.8 77.7 84c11-14.4 23.5-30.1 37-42.9c7.9-7.4 20.1-7.4 28 .1c34.6 33 63.9 76.6 84.5 118c20.3 40.8 33.8 82.5 33.8 111.9C448 404.2 348.2 512 224 512C98.4 512 0 404.1 0 276.5c0-38.4 17.8-85.3 45.4-131.7C73.3 97.7 112.7 48.6 159.3 5.4zM225.7 416c25.3 0 47.7-7 68.8-21c42.1-29.4 53.4-88.2 28.1-134.4c-4.5-9-16-9.6-22.5-2l-25.2 29.3c-6.6 7.6-18.5 7.4-24.7-.5c-16.5-21-46-58.5-62.8-79.8c-6.3-8-18.3-8.1-24.7-.1c-33.8 42.5-50.8 69.3-50.8 99.4C112 375.4 162.6 416 225.7 416z"/></svg>';
        
        // Icono Check (Aparece al completar)
        $icon_check = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" style="display:inline-block; width:1em; height:1em; vertical-align:-0.125em; fill:currentColor; margin-right:6px;"><path d="M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM369 209L241 337c-9.4 9.4-24.6 9.4-33.9 0l-64-64c-9.4-9.4-9.4-24.6 0-33.9s24.6-9.4 33.9 0l47 47L335 175c9.4-9.4 24.6-9.4 33.9 0s9.4 24.6 0 33.9z"/></svg>';

        // Helper para formatear el precio con el estilo
        $format_price = function( $val ) use ( $style_amount ) {
            return '<span style="' . esc_attr( $style_amount ) . '">' . wc_price( $val ) . '</span>';
        };

        // Lógica de Mensaje + Color
        // NOTA: Usamos wp_kses_post() en los mensajes de usuario AHORA, antes de concatenar el SVG
        if ( $total == 0 ) {
            // Estado 1: Inicial
            $safe_msg = wp_kses_post( $msg_initial );
            $message = str_replace( '{amount}', $format_price( $threshold ), $safe_msg );
            $current_bg_color = $bg_initial;
        } elseif ( $missing > 0 ) {
            // Estado 2: Progreso
            $safe_msg = wp_kses_post( $msg_progress );
            $message = str_replace( '{amount}', $format_price( $missing ), $safe_msg ) . $icon_fire;
            $current_bg_color = $bg_progress;
        } else {
            // Estado 3: Completado
            $safe_msg = wp_kses_post( $msg_completed );
            $message = $icon_check . $safe_msg;
            $current_bg_color = $bg_completed;
        }

        // Construcción de estilos
        $styles = "background-color: {$current_bg_color} !important; color: {$text_color} !important; text-align: center !important; padding: 15px !important; z-index: 9999 !important; box-shadow: 0 2px 5px rgba(0,0,0,0.1) !important; transition: all 0.3s ease !important; width: 100% !important; box-sizing: border-box !important;";
        
        $is_fixed = ( $display_method === 'automatic' && ! $is_shortcode );
        
        if ( $is_fixed ) {
            $styles .= " position: fixed !important; left: 0 !important;";
            if ( $position === 'bottom' ) {
                $styles .= " bottom: 0 !important; top: auto !important; box-shadow: 0 -2px 5px rgba(0,0,0,0.1) !important;"; 
            } else {
                $styles .= " top: 0 !important; bottom: auto !important;";
            }
        }

        ob_start();
        ?>
        <div id="sfsb-bar" style="<?php echo esc_attr( $styles ); ?>">
            <span class="sfsb-message" style="font-size: 16px; font-weight: 500; display: flex; align-items: center; justify-content: center; flex-wrap: wrap;">
                <?php 
                // Imprimimos el mensaje directamente para respetar los SVGs.
                // El contenido de usuario ($msg_...) ya ha sido sanitizado arriba con wp_kses_post().
                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
                echo $message; 
                ?>
            </span>
        </div>
        <?php if ( $is_fixed && $position === 'top' ) : ?>
        <style>
            body.admin-bar #sfsb-bar { top: 32px !important; }
            @media screen and (max-width: 782px) {
                body.admin-bar #sfsb-bar { top: 46px !important; }
            }
            body { margin-top: 54px !important; } 
            
            /* Animación de pulso para el icono de fuego */
            @keyframes sfsb-pulse {
                0%, 100% { transform: scale(1); opacity: 0.9; }
                50% { transform: scale(1.1); opacity: 1; }
            }
            #sfsb-bar svg { animation: sfsb-pulse 2s infinite ease-in-out; }
        </style>
        <?php endif; ?>
        <?php if ( $is_fixed && $position === 'bottom' ) : ?>
        <style>
            body { margin-bottom: 54px !important; } 
        </style>
        <?php endif; ?>
        <?php
        return ob_get_clean();
    }

    /**
     * Render the notification bar in the frontend.
     */
    public function render_bar() {
        $display_method = get_option( 'sfsb_display_method', 'automatic' );
        if ( $display_method === 'automatic' ) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            echo $this->get_bar_html();
        }
    }
    
    /**
     * Render Pro Feature Banner.
     */
    public function render_pro_banner() {
        ?>
        <div class="sfsb-pro-banner" style="background: #f0f0f1; border-left: 4px solid #72aee6; padding: 12px; margin-bottom: 20px;">
            <p style="margin: 0; display: inline-block;">
                <?php esc_html_e( 'Want more features like Geolocation and Scheduling?', 'simple-free-shipping-bar-ocma' ); ?>
            </p>
            <a href="<?php echo esc_url( SFSB_PRO_URL ); ?>" target="_blank" class="button button-primary" style="float: right;">
                <?php esc_html_e( 'Upgrade to Pro', 'simple-free-shipping-bar-ocma' ); ?>
            </a>
            <div style="clear: both;"></div>
        </div>
        <?php
    }

    /**
     * Update fragments via AJAX.
     *
     * @param array $fragments
     * @return array
     */
    public function update_fragments( $fragments ) {
        $fragments['div#sfsb-bar'] = $this->get_bar_html();
        return $fragments;
    }


    /**
     * Render the Premium Upsell Page.
     */
    public function render_premium_upsell_page() {
        // Check if Pro plugin class exists to determine CTA
        $pro_installed = class_exists( 'Simple_Free_Shipping_Bar_Pro_Ocma' );
        
        ?>
        <div class="sfsb-upsell-container" style="max-width: 900px; margin: auto; margin-top: 30px;">
            <div style="text-align: center; margin-bottom: 40px;">
                <h2 style="font-size: 2.5em; margin-bottom: 10px;"><?php esc_html_e( 'Unlock the Full Power of Free Shipping', 'simple-free-shipping-bar-ocma' ); ?></h2>
                <p style="font-size: 1.2em; color: #666;"><?php esc_html_e( 'Upgrade to Pro to access advanced targeting and scheduling features.', 'simple-free-shipping-bar-ocma' ); ?></p>
            </div>

            <div class="sfsb-upsell-features" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 30px; margin-bottom: 40px;">
                
                <!-- Feature 1: Geolocation -->
                <div class="sfsb-feature-card" style="background: #fff; border: 1px solid #e5e7eb; border-radius: 12px; padding: 30px; box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);">
                    <div class="sfsb-feature-icon" style="background: #e0e7ff; color: #4338ca; width: 60px; height: 60px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 20px;">
                        <span class="dashicons dashicons-admin-site" style="font-size: 32px; width: 28px; height: 28px;"></span>
                    </div>
                    <h3 style="margin-top: 0; font-size: 1.4em;"><?php esc_html_e( 'Geolocation Targeting', 'simple-free-shipping-bar-ocma' ); ?></h3>
                    <p style="color: #4b5563; line-height: 1.6;"><?php esc_html_e( 'Show specific free shipping goals to customers based on their country. Perfect for international stores with different shipping rates.', 'simple-free-shipping-bar-ocma' ); ?></p>
                </div>

                <!-- Feature 2: Scheduling -->
                <div class="sfsb-feature-card" style="background: #fff; border: 1px solid #e5e7eb; border-radius: 12px; padding: 30px; box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);">
                    <div class="sfsb-feature-icon" style="background: #ecfdf5; color: #047857; width: 60px; height: 60px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-bottom: 20px;">
                        <span class="dashicons dashicons-calendar-alt" style="font-size: 32px; width: 32px; height: 32px;"></span>
                    </div>
                    <h3 style="margin-top: 0; font-size: 1.4em;"><?php esc_html_e( 'Advanced Scheduling', 'simple-free-shipping-bar-ocma' ); ?></h3>
                    <p style="color: #4b5563; line-height: 1.6;"><?php esc_html_e( 'Plan your promotions in advance. Set start and end dates for your shipping campaigns to run automatically.', 'simple-free-shipping-bar-ocma' ); ?></p>
                </div>

            </div>

            <div style="text-align: center;">
                <?php if ( $pro_installed ) : ?>
                    <a href="?page=simple-free-shipping-bar-ocma&tab=license" class="button button-primary button-hero">
                        <?php esc_html_e( 'Activate License', 'simple-free-shipping-bar-ocma' ); ?>
                        <span class="dashicons dashicons-arrow-right-alt" style="margin-top: 12px; margin-left: 5px;"></span>
                    </a>
                    <p class="description" style="margin-top: 10px;"><?php esc_html_e( 'Pro plugin is installed. Please activate your license.', 'simple-free-shipping-bar-ocma' ); ?></p>
                <?php else : ?>
                    <a href="<?php echo esc_url( SFSB_PRO_URL ); ?>" target="_blank" class="button button-primary button-hero">
                        <?php esc_html_e( 'Upgrade to Pro', 'simple-free-shipping-bar-ocma' ); ?>
                        <span class="dashicons dashicons-external" style="margin-top: 10px; margin-left: 5px;"></span>
                    </a>
                    <p class="description" style="margin-top: 10px;"><?php esc_html_e( 'Get instant access to all premium features.', 'simple-free-shipping-bar-ocma' ); ?></p>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }

}

// Initialize the plugin
add_action( 'plugins_loaded', array( 'Simple_Free_Shipping_Bar', 'get_instance' ) );
